/*____________________________________________________________________________
		Copyright (C) 2000 Network Associates, Inc.
        All rights reserved.

        $Id: CRegistryKey.cpp,v 1.2 1999/10/06 06:41:00 nryan Exp $
____________________________________________________________________________*/

#include "pgpClassesConfig.h"

#include "CArray.h"
#include "CRegistryKey.h"
#include "CString.h"

_USING_PGP

// Class CRegistryKey member functions

CRegistryKey::~CRegistryKey()
{
	try
	{
		if (IsOpened())
			Close();
	}
	catch (CComboError&) { }
}

void 
CRegistryKey::Open(HKEY root, const char *section, REGSAM security)
{
	pgpAssert(!IsOpened());
	pgpAssertStrValid(section);
	
	PGPUInt32	result	= RegOpenKeyEx(root, section, 0, security, &mHKey);

	if (result != ERROR_SUCCESS)
		THROW_ERRORS(kPGPError_Win32RegistryOpFailed, result);

	mIsOpened = TRUE;
}

void 
CRegistryKey::Create(HKEY root, const char *section, REGSAM security)
{
	pgpAssert(!IsOpened());
	pgpAssertStrValid(section);

	PGPUInt32		result;
	unsigned long	disp;
	
	result = RegCreateKeyEx(root, section, 0, NULL, 0, security, NULL, 
		&mHKey, &disp);

	if (result != ERROR_SUCCESS)
		THROW_ERRORS(kPGPError_Win32RegistryOpFailed, result);

	mIsOpened = TRUE;
}

void 
CRegistryKey::Close()
{
	pgpAssert(IsOpened());

	RegCloseKey(mHKey);
	mHKey = NULL;

	mIsOpened = FALSE;
}

PGPUInt32 
CRegistryKey::GetValue(
	const char	*valueName, 
	PGPByte		*buf, 
	PGPUInt32	sizeBuf) const
{
	pgpAssert(IsOpened());
	pgpAssertStrValid(valueName);
	pgpAssertAddrValid(buf, VoidAlign);
	
	PGPUInt32	dataType, result;

	result = RegQueryValueEx(mHKey, valueName, NULL, 
		reinterpret_cast<unsigned long *>(&dataType), buf, 
		reinterpret_cast<unsigned long *>(&sizeBuf));

	if (result != ERROR_SUCCESS)
		THROW_ERRORS(kPGPError_Win32RegistryOpFailed, result);

	return sizeBuf;
}

PGPBoolean 
CRegistryKey::EnumValues(
	PGPUInt32	index, 
	CString&	valueName, 
	PGPUInt32&	valueType) const
{
	pgpAssert(IsOpened());
	pgpAssertAddrValid(valueName, char);
	
	PGPBoolean	isIndexValid	= TRUE;
	PGPUInt32	result;
	PGPUInt32	sizeValueName	= CString::kDefaultStringSize;

	result = RegEnumValue(mHKey, index, valueName.GetBuffer(sizeValueName), 
		reinterpret_cast<unsigned long *>(&sizeValueName), NULL, 
		reinterpret_cast<unsigned long *>(&valueType), NULL, NULL);
	valueName.ReleaseBuffer();

	if (result != ERROR_SUCCESS)
	{
		if (result == ERROR_NO_MORE_ITEMS)
			isIndexValid = FALSE;
		else
			THROW_ERRORS(kPGPError_Win32RegistryOpFailed, result);
	}

	return isIndexValid;
}

PGPBoolean 
CRegistryKey::EnumSubkeys(
	PGPUInt32	index, 
	CString&	subkeyName) const
{
	pgpAssert(IsOpened());
	pgpAssertAddrValid(subkeyName, char);

	PGPBoolean	isIndexValid	= TRUE;
	PGPUInt32	result;
	PGPUInt32	sizeSubkeyName	= CString::kDefaultStringSize;

	result = RegEnumKeyEx(mHKey, index, 
		subkeyName.GetBuffer(sizeSubkeyName), 
		reinterpret_cast<unsigned long *>(&sizeSubkeyName), NULL, NULL, 
		NULL, NULL);
	subkeyName.ReleaseBuffer();

	if (result != ERROR_SUCCESS)
	{
		if (result == ERROR_NO_MORE_ITEMS)
			isIndexValid = FALSE;
		else
			THROW_ERRORS(kPGPError_Win32RegistryOpFailed, result);
	}

	return isIndexValid;
}

void 
CRegistryKey::SetValue(
	const char		*valueName, 
	const PGPByte	*data, 
	PGPUInt32		sizeData, 
	PGPUInt32		dataType) const
{
	pgpAssert(IsOpened());
	pgpAssertAddrValid(data, VoidAlign);
	
	PGPUInt32	result	= RegSetValueEx(mHKey, valueName, 0, dataType, data, 
		sizeData);

	if (result != ERROR_SUCCESS)
		THROW_ERRORS(kPGPError_Win32RegistryOpFailed, result);
}

void 
CRegistryKey::DeleteValue(const char *valueName) const
{
	pgpAssert(IsOpened());
	
	PGPUInt32	result	= RegDeleteValue(mHKey, valueName);

	if (result != ERROR_SUCCESS)
		THROW_ERRORS(kPGPError_Win32RegistryOpFailed, result);
}

void 
CRegistryKey::DeleteSubkey(const char *subkeyName, PGPBoolean recursive) const
{
	pgpAssert(IsOpened());
	pgpAssertStrValid(subkeyName);

	CRegistryKey	subkey;
	subkey.Open(mHKey, subkeyName);

	if (recursive)
	{
		while (TRUE)
		{
			CString		keyName;

			if (!subkey.EnumSubkeys(0, keyName))
				break;

			subkey.DeleteSubkey(keyName, TRUE);
		}
	}

	PGPUInt32	result	= RegDeleteKey(mHKey, subkeyName);

	if (result != ERROR_SUCCESS)
		THROW_ERRORS(kPGPError_Win32RegistryOpFailed, result);
}
